using System;
using System.Collections;
using System.Data;
using System.Drawing.Printing;
using System.Reflection;

namespace gov.va.med.vbecs.BOL
{
	/// <summary>
	/// Summary description for DivisionAlerts.
	/// </summary>
	public class DivisionAlerts : BaseBusinessObject
	{
		#region variables

		// printing
		private bool	_printAlertsIndicator;
		private string	_printerName;
		private Common.OrderAlertTypeCode _printOrderAlertType;
		private int		_printOrderNeededHours;

		// icon
		private bool	_iconAlertsIndicator;
		private Common.OrderAlertTypeCode _iconOrderAlertType;
		private int		_iconOrdersNeededHours;
		private int		_iconRefreshRate;
		
		// login message
		private string _loginMessage;
		
		#endregion


		#region Constructors

		///<Developers>
		///	<Developer>Carl Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/13/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3622"> 
		///		<ExpectedInput>DataRow with valid data</ExpectedInput>
		///		<ExpectedOutput>DivisionAlerts object with properties set to the data in 
		///		the DataRow.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3623"> 
		///		<ExpectedInput>DataRow with no data</ExpectedInput>
		///		<ExpectedOutput>DivisionAlerts object with properties default values.</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// This constructor takes a DataRow and instantiates the object.  If there is
		/// no data, some of the properties are set to default values in accordance with 
		/// the business rules.  
		/// </summary>
		public DivisionAlerts(DataRow dtRow)
		{
			this.LoadFromDataRow(dtRow);
		}

		#endregion


		#region Properties

		#region Printer

		///<Developers>
		///	<Developer>Carl Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/13/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3626"> 
		///		<ExpectedInput>true</ExpectedInput>
		///		<ExpectedOutput>Value of true</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3627"> 
		///		<ExpectedInput>true</ExpectedInput>
		///		<ExpectedOutput>Broken rule set because other data not set correctly.</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Implements BR_95.08
		/// </summary>
		public bool PrintAlertsIndicator
		{
			set
			{
				this._printAlertsIndicator = value;
				this.IsDirty = true;

				// Setting this indicator to true will require that the Printer Name) and Order Alert Type 
				// fields are set to true, so those broken rules are called.
				// BR_95.08
				RuleBroken("PrinterName", (value == true && this._printerName == String.Empty ));
				RuleBroken("PrintOrderAlertType", (value == true && this._printOrderAlertType == Common.OrderAlertTypeCode.Unknown ));
				RuleBroken("PrintOrderNeededHours", (value == true && this._printOrderAlertType == Common.OrderAlertTypeCode.Hour_limit && this._printOrderNeededHours == 0));
			}
			get
			{
				return this._printAlertsIndicator;
			}
		}

		///<Developers>
		///	<Developer>Carl Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/13/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3624"> 
		///		<ExpectedInput>String of length > 0</ExpectedInput>
		///		<ExpectedOutput>Valid printer name</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3625"> 
		///		<ExpectedInput>Empty String</ExpectedInput>
		///		<ExpectedOutput>Broken rule set.</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Printer Name
		/// BR_95.08
		/// </summary>
		public string PrinterName
		{
			set
			{
				this._printerName = value;
				this.IsDirty = true;
				RuleBroken(MethodBase.GetCurrentMethod().Name, (value.Trim().Length == 0 && this._printAlertsIndicator == true));
			}
			get
			{
				return this._printerName;
			}
		}

		///<Developers>
		///	<Developer>Carl Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/13/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3628"> 
		///		<ExpectedInput>Order alert type that is not unknown</ExpectedInput>
		///		<ExpectedOutput>Valid Order alerty type</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3629"> 
		///		<ExpectedInput>Order alert type set to unknown</ExpectedInput>
		///		<ExpectedOutput>Broken rule set.</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Implements BR_95.08
		/// </summary>
		public Common.OrderAlertTypeCode PrintOrderAlertType
		{
			set
			{
				this._printOrderAlertType = value;
				this.IsDirty = true;
				RuleBroken(MethodBase.GetCurrentMethod().Name, value == Common.OrderAlertTypeCode.Unknown && this._printAlertsIndicator == true);
				// BR_95.05 - If this field is set to hour limit, the Needed Hours field must be set to a value other than 0.
				RuleBroken("PrintOrderNeededHours", value == Common.OrderAlertTypeCode.Hour_limit && this._printOrderNeededHours == 0 && this._printAlertsIndicator == true);
			}
			get
			{
				return this._printOrderAlertType;
			}
		}

		///<Developers>
		///	<Developer>Carl Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/13/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3630"> 
		///		<ExpectedInput>A integer > 0</ExpectedInput>
		///		<ExpectedOutput>Valid Print order needed hours</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3631"> 
		///		<ExpectedInput>0</ExpectedInput>
		///		<ExpectedOutput>Broken rule set.</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// PrintOrderNeededHours
		/// </summary>
		public int PrintOrderNeededHours
		{
			set
			{
				this._printOrderNeededHours = value;
				this.IsDirty = true;
				RuleBroken(MethodBase.GetCurrentMethod().Name, (value == 0 && this._printOrderAlertType == Common.OrderAlertTypeCode.Hour_limit && this._printAlertsIndicator == true));
			}
			get
			{
				return this._printOrderNeededHours;
			}
		}

		#endregion

		#region Icon

		///<Developers>
		///	<Developer>Carl Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/13/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3632"> 
		///		<ExpectedInput>true</ExpectedInput>
		///		<ExpectedOutput>Value of true</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3633"> 
		///		<ExpectedInput>true</ExpectedInput>
		///		<ExpectedOutput>Broken rule set because other data not set correctly..</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// IconAlertsIndicator
		/// </summary>
		public bool IconAlertsIndicator
		{
			set
			{
				this._iconAlertsIndicator = value;
				this.IsDirty = true;
				// Setting this indicator to true will require that the Refresh Rate and Order Alert Type 
				// fields are set to true, so those broken rules are called.
				RuleBroken("IconOrderAlertType", (value == true && this._iconOrderAlertType == Common.OrderAlertTypeCode.Unknown ));
				RuleBroken("IconOrdersNeededHours", (value == true && this._iconOrderAlertType == Common.OrderAlertTypeCode.Hour_limit && this._iconOrdersNeededHours == 0));
				RuleBroken("IconRefreshRate", (value == true && ((this._iconRefreshRate < Convert.ToInt32(Common.OrderAlertRefreshRate.Minimum)) || (this._iconRefreshRate > Convert.ToInt32(Common.OrderAlertRefreshRate.Maximum)))));
			}
			get
			{
				return this._iconAlertsIndicator;
			}
		}

		///<Developers>
		///	<Developer>Carl Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/13/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3638"> 
		///		<ExpectedInput>An integer > 0</ExpectedInput>
		///		<ExpectedOutput>Valid Icon refresh rate</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3639"> 
		///		<ExpectedInput>0</ExpectedInput>
		///		<ExpectedOutput>Broken rule set</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// IconRefreshRate
		/// </summary>
		public int IconRefreshRate
		{
			set
			{
				this._iconRefreshRate = value;
				this.IsDirty = true;
				RuleBroken(MethodBase.GetCurrentMethod().Name, (this._iconAlertsIndicator == true && ((value < Convert.ToInt32(Common.OrderAlertRefreshRate.Minimum)) || (value > Convert.ToInt32(Common.OrderAlertRefreshRate.Maximum)))));
			}
			get
			{
				return this._iconRefreshRate;
			}
		}

		///<Developers>
		///	<Developer>Carl Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/13/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3634"> 
		///		<ExpectedInput>An Order alert type other than Unknown</ExpectedInput>
		///		<ExpectedOutput>Valid icon order alert type</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3635"> 
		///		<ExpectedInput>An Order alert type of Nnknown</ExpectedInput>
		///		<ExpectedOutput>Broken rule set</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// IconOrderAlertType
		/// </summary>
		public Common.OrderAlertTypeCode IconOrderAlertType
		{
			set
			{
				this._iconOrderAlertType = value;
				this.IsDirty = true;
				RuleBroken(MethodBase.GetCurrentMethod().Name, (value == Common.OrderAlertTypeCode.Unknown && this._iconAlertsIndicator == true ));
				// BR_95.05 - If this field is set to hour limit, the Needed Hours field must be set to a value other than 0.
				RuleBroken("IconOrdersNeededHours", value == Common.OrderAlertTypeCode.Hour_limit && this._iconOrdersNeededHours == 0 && this._iconAlertsIndicator == true);

			}
			get
			{
				return this._iconOrderAlertType;
			}
		}

		///<Developers>
		///	<Developer>Carl Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/13/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3636"> 
		///		<ExpectedInput>Integer > 0</ExpectedInput>
		///		<ExpectedOutput>Valid Icon orders needed</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3637"> 
		///		<ExpectedInput>0</ExpectedInput>
		///		<ExpectedOutput>Broken rule set</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// IconOrdersNeededHours
		/// </summary>
		public int IconOrdersNeededHours
		{
			set
			{
				this._iconOrdersNeededHours = value;
				this.IsDirty = true;
				RuleBroken(MethodBase.GetCurrentMethod().Name, (value == 0 && this._iconOrderAlertType == Common.OrderAlertTypeCode.Hour_limit && this._iconAlertsIndicator == true ));
			}
			get
			{
				return this._iconOrdersNeededHours;
			}
		}

		#endregion

		///<Developers>
		///	<Developer>Carl Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/14/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3642"> 
		///		<ExpectedInput>string</ExpectedInput>
		///		<ExpectedOutput>string</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3643"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// LoginMessage
		/// </summary>
		public string LoginMessage
		{
			set
			{
				this._loginMessage = value;
				this.IsDirty = true;
			}
			get
			{
			return this._loginMessage;
			}
		}

		#endregion


		#region Methods

		///<Developers>
		///	<Developer>Carl Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/28/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3671"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>DataTable</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="3672"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// GetOrderAlertTypeList
		/// </summary>
		/// <returns></returns>
		public static DataTable GetOrderAlertTypeList()
		{
			return DAL.DivisionAlerts.GetOrderAlertTypeList();
		}

		///<Developers>
		///	<Developer>Carl Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/14/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3640"> 
		///		<ExpectedInput>None</ExpectedInput>
		///		<ExpectedOutput>ArrayList of printers configured for the current machine.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3641"> 
		///		<ExpectedInput>None</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// This method returns an ArrayList of strings representing each printer currently configured on 
		/// the machine.  The full UNC name is returned in the string: \\Machine name\printer name .
		/// BR_95.03
		/// </summary>
		public static ArrayList GetPrinters()
		{
			ArrayList printerList = new ArrayList();
		
			foreach(String printer in PrinterSettings.InstalledPrinters)
			{
				printerList.Add(printer);
			}
            return printerList;
		}

		/// <summary>
		/// LoadFromDataRow
		/// </summary>
		/// <param name="dtRow"></param>
		protected override void LoadFromDataRow(DataRow dtRow)
		{
			// Icon
			if (dtRow.Table.Columns.Contains(Common.VbecsTables.VamcDivision.IconAlertsIndicator))
			{
				// BR_95.06 - Alerts are turned off by default
				this._iconAlertsIndicator = dtRow.IsNull(Common.VbecsTables.VamcDivision.IconAlertsIndicator) ? false : (bool) dtRow[Common.VbecsTables.VamcDivision.IconAlertsIndicator];
			}
			if (dtRow.Table.Columns.Contains(Common.VbecsTables.VamcDivision.IconOrderAlertTypeCode))
			{
				this._iconOrderAlertType = dtRow.IsNull(Common.VbecsTables.VamcDivision.IconOrderAlertTypeCode) ? Common.OrderAlertTypeCode.Unknown :  Common.Utility.GetOrderAlertTypeTypeFromString(dtRow[Common.VbecsTables.VamcDivision.IconOrderAlertTypeCode].ToString());
			}
			if (dtRow.Table.Columns.Contains(Common.VbecsTables.VamcDivision.IconOrdersNeededHours))
			{
				// BR_95.05 - set to 48 by default
				this._iconOrdersNeededHours = dtRow.IsNull(Common.VbecsTables.VamcDivision.IconOrdersNeededHours) ? 48 : System.Convert.ToInt32(dtRow[Common.VbecsTables.VamcDivision.IconOrdersNeededHours]);
			}
			if (dtRow.Table.Columns.Contains(Common.VbecsTables.VamcDivision.IconRefreshRate))
			{
				// BR_95.07 - set to 60 by default 
				this._iconRefreshRate = dtRow.IsNull(Common.VbecsTables.VamcDivision.IconRefreshRate) ? Convert.ToInt32(Common.OrderAlertRefreshRate.Default) :  Convert.ToInt32(dtRow[Common.VbecsTables.VamcDivision.IconRefreshRate].ToString());
			}

			// Printer
			if (dtRow.Table.Columns.Contains(Common.VbecsTables.VamcDivision.PrinterAlertsIndicator))
			{
				// BR_95.06 - Alerts are turned off by default
				this._printAlertsIndicator = dtRow.IsNull(Common.VbecsTables.VamcDivision.PrinterAlertsIndicator) ? false : (bool) dtRow[Common.VbecsTables.VamcDivision.PrinterAlertsIndicator];
			}
			if (dtRow.Table.Columns.Contains(Common.VbecsTables.VamcDivision.PrinterOrderAlertTypeCode))
			{
				this._printOrderAlertType = dtRow.IsNull(Common.VbecsTables.VamcDivision.PrinterOrderAlertTypeCode) ? Common.OrderAlertTypeCode.Unknown :  Common.Utility.GetOrderAlertTypeTypeFromString(dtRow[Common.VbecsTables.VamcDivision.PrinterOrderAlertTypeCode].ToString());
			}
			if (dtRow.Table.Columns.Contains(Common.VbecsTables.VamcDivision.PrinterOrdersNeededHours))
			{
				// BR_95.05 - set to 48 by default
				this._printOrderNeededHours = dtRow.IsNull(Common.VbecsTables.VamcDivision.PrinterOrdersNeededHours) ? 48 : System.Convert.ToInt32(dtRow[Common.VbecsTables.VamcDivision.PrinterOrdersNeededHours]);
			}
			// ToDo: Uncommment this once Bob has added the column to the table.
			if (dtRow.Table.Columns.Contains(Common.VbecsTables.VamcDivision.PrinterAlertsPrinterName))
			{
				this._printerName = dtRow.IsNull(Common.VbecsTables.VamcDivision.PrinterAlertsPrinterName) ? String.Empty : dtRow[Common.VbecsTables.VamcDivision.PrinterAlertsPrinterName].ToString();
			}

			// Login message
			// ToDo: Uncommment this once Bob has added the column to the table.
			if (dtRow.Table.Columns.Contains(Common.VbecsTables.VamcDivision.LoginMessageText))
			{
				this._loginMessage = dtRow.IsNull(Common.VbecsTables.VamcDivision.LoginMessageText) ? String.Empty : dtRow[Common.VbecsTables.VamcDivision.LoginMessageText].ToString();
			}

			this.IsDirty = false;
			this.IsNew = false;
		}

		///<Developers>
		///	<Developer></Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/8/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="7447"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="7448"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// LoadDataRowFromThis
		/// </summary>
		/// <param name="dtRow"></param>
		/// <returns></returns>
		public override DataRow LoadDataRowFromThis(DataRow dtRow) 
		{
			return dtRow;
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>9/14/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6009"> 
		///		<ExpectedInput>Order alert type of All Orders</ExpectedInput>
		///		<ExpectedOutput>True</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="8155"> 
		///		<ExpectedInput>Order alert type of Specimen OK or NA</ExpectedInput>
		///		<ExpectedOutput>True</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="0" testid ="8156"> 
		///		<ExpectedInput>Order alert type of Hour Limit</ExpectedInput>
		///		<ExpectedOutput>True</ExpectedOutput>
		///	</Case>
		///
		///
		///
		///<Case type="1" testid ="6010"> 
		///		<ExpectedInput>Invalid order alert type</ExpectedInput>
		///		<ExpectedOutput>False</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// IsPatientOrderPrintAlertRequired
		/// Implements BR_84.12
		/// </summary>
		/// <param name="patientOrder"></param>
		/// <param name="orderedComponents"></param>
		/// <param name="orderedTests"></param>
		/// <param name="divisionCode"></param>
		/// <returns></returns>
		public bool IsPatientOrderPrintAlertRequired(DataTable patientOrder, DataTable orderedComponents, DataTable orderedTests, string divisionCode)
		{
			DateTime currDateTime = BOL.VBECSDateTime.GetDivisionCurrentDateTime(divisionCode);
			//
			if (this._printOrderAlertType == Common.OrderAlertTypeCode.All_orders)
			{
				return true;
			}
			//
			if (this._printOrderAlertType == Common.OrderAlertTypeCode.Specimen_OK_or_NA)
			{
				foreach(DataRow orderedComponent in orderedComponents.Rows)
				{
					if ((orderedComponent[Common.VbecsTables.SpecimenStatus.SpecimenStatusCode].ToString() == Common.SpecimenStatus.NSR.ToString()) || !orderedComponent.IsNull(Common.VbecsTables.PatientSpecimen.PatientSpecimenGuid))
					{
						return true;
					}

				}
				//
				foreach(DataRow orderedTest in orderedTests.Rows)
				{
					if (!orderedTest.IsNull(Common.VbecsTables.PatientSpecimen.PatientSpecimenGuid))
					{
						return true;
					}

				}
			}
			//
			if (this._printOrderAlertType == Common.OrderAlertTypeCode.Hour_limit)
			{
				foreach(DataRow orderedComponent in orderedComponents.Rows)
				{
					if (!orderedComponent.IsNull(Common.VbecsTables.OrderedComponent.RequiredDatetime))
					{
						if (Convert.ToDateTime(orderedComponent[Common.VbecsTables.OrderedComponent.RequiredDatetime]) <= currDateTime.AddHours(this._printOrderNeededHours))
						{
							return true;
						}
					}
				}
				//
				foreach(DataRow orderedTest in orderedTests.Rows)
				{
					if (!orderedTest.IsNull(Common.VbecsTables.OrderedTest.RequiredResultsDate))
					{
						if (Convert.ToDateTime(orderedTest[Common.VbecsTables.OrderedTest.RequiredResultsDate]) <= currDateTime.AddHours(this._printOrderNeededHours))
						{
							return true;
						}
					}
				}
			}
			//
			return false;
		}

		#endregion
	}
}
